
#ifndef system_class_h
#define system_class_h

#include <random>
#include <array>
#include <iostream>
#include <vector>
#include <thread>
#include <H5Cpp.h>
#include <csignal>
#include <string>

//implemented as template because of data structures chosen
template <int ring_size,int number_of_species,int critical_size>
/// Base class for both simulation types (purely virtual)
class RingStructureBaseClass {
    
    public:
    
    /// Base Class constructor (virtual)
    /// @param initial_number_of_particles initial number of particle in the inactive state
    /// @param activation_rate activation rate
    /// @param reaction_rates binding rates for all different polymer sizes
    /// @param decay_rates decay rates for all subcritical ploymers
    /// @param record_evolution bool for recording of intermediate states
    /// @param time_between_recordings time between recordings
    /// @param seed seed for the random number generator (has nothing to do with nucleation)
    RingStructureBaseClass(const long initial_number_of_particles,const double activation_rate,const std::array<double, ring_size-1> reaction_rates,const std::array<double, critical_size-2> decay_rates,const bool record_evolution,const double time_between_recordings,const long seed)://Begin List
    time_{0},
    accumulated_decay_rate_{0},
    activation_rate_{activation_rate},
    reaction_rates_{reaction_rates},
    decay_rates_{decay_rates},
    decay_structures_{0},
    record_time_evolution_{record_evolution},
    time_between_recordings_{time_between_recordings},
    initial_number_of_particles_{initial_number_of_particles},
    rng_engine_(seed)//End List
    {//Begin Function
        maximal_number_of_ring_structures_=(initial_number_of_particles*number_of_species)/ring_size;
        if (record_evolution) {
            time_trace_.push_back(0);
            next_recording_=time_between_recordings_;
        }
        if ((initial_number_of_particles*number_of_species)%(ring_size)) {
            std::cout<<"Fatal Error! Simulation may not terminate."<<std::endl;
            std::raise(SIGTERM);
            exit(SIGTERM);
        }
        
    };//End Constructor
    
    /// Copy constructor
    /// @param original other system
    /// @param seed seed for the random number generator for different time evolution (has nothing to do with nucleation)
    RingStructureBaseClass(const RingStructureBaseClass& original,const long seed):
    record_time_evolution_{original.record_time_evolution_},
    time_between_recordings_{original.time_between_recordings_},
    next_recording_{original.next_recording_},
    time_{original.time_},
    activation_rate_{original.activation_rate_},
    accumulated_decay_rate_{original.accumulated_decay_rate_},
    maximal_number_of_ring_structures_{original.maximal_number_of_ring_structures_},
    decay_structures_{original.decay_structures_},
    reaction_rates_{original.reaction_rates_},
    decay_rates_{original.decay_rates_},
    accumulated_rates_{original.accumulated_rates_},
    time_trace_{original.time_trace_},
    initial_number_of_particles_{original.initial_number_of_particles_},
    rng_engine_(seed)
    {}//End Copy Constructor
    
    
    /// update time according to Gillespie
    inline void UpdateTime(){
        time_+=std::exponential_distribution<double>(accumulated_rates_.back())(rng_engine_);
    }

    /// calculate rates
    virtual void GetAccumulatedRates()=0;

    /// record intermediate states
    virtual void RecordTimeTrace()=0;

    /// choose event according to Gillespie
    virtual void ChooseEvent()=0;
    
    /// perform single iteration step
    /// @return bool deciding if simulation will be continued
    const bool IterationStep(){
        
        GetAccumulatedRates();
        
        if (accumulated_rates_.back()) {//Reaction takes place
            UpdateTime();
            if (record_time_evolution_&&time_>next_recording_) {
                RecordTimeTrace();
            }
            ChooseEvent();
            return true;
        }//Reaction takes place
        
        else return false;//final state
        
    }

    
    /// write final results to vector
    /// @param time_trace pointer to vector for storage
    void GetResults(std::vector<double>* time_trace){
        *time_trace=time_trace_;
    }

    /// perform simulation of a copy
    /// @param seed for the random number generator for different time evolution (has nothing to do with nucleation)
    /// @param time_trace pointer to vector for storage
    virtual void operator()(unsigned seed,std::vector<double>* time_trace)=0;

    
    /// Write parameters to an HDF5 group
    /// @param group hdf5 group
    void WriteReactionAndDecayParametersToHDF5Group(H5::Group& group){
        hsize_t dims1=ring_size-1;
        H5::DataSpace dataspace1(1,&dims1);
        H5::DataSet dataset1=group.createDataSet("reaction_parameters", H5::PredType::NATIVE_DOUBLE, dataspace1);
        dataset1.write(reaction_rates_.data(), H5::PredType::NATIVE_DOUBLE);
        
        if (0<decay_rates_.size()) {
            hsize_t dims2=decay_rates_.size();
            H5::DataSpace dataspace2(1,&dims2);
            H5::DataSet dataset2=group.createDataSet("decay_parameters", H5::PredType::NATIVE_DOUBLE, dataspace2);
            dataset2.write(decay_rates_.data(), H5::PredType::NATIVE_DOUBLE);
        }
    }

    
    /// Write all attributes to an HDF5 group
    /// @param group hdf5 group
    void AttachAttributesToHDF5Group(H5::Group& group){
        int tmp_number_ofspecies=number_of_species;
        int tmp_ring_size=ring_size;
        int tmp_critical_ring_size=critical_size;
        hsize_t dims[]={1};
        H5::DataSpace attr=H5::DataSpace(1,dims);
        H5::Attribute attribute[]={group.createAttribute("initial particle number",H5::PredType::NATIVE_LONG,attr),group.createAttribute("number of species",H5::PredType::NATIVE_INT,attr),group.createAttribute("ring size",H5::PredType::NATIVE_INT,attr),group.createAttribute("critical size",H5::PredType::NATIVE_INT,attr),group.createAttribute("activation rate",H5::PredType::NATIVE_DOUBLE,attr)};
        attribute[0].write(H5::PredType::NATIVE_LONG,&initial_number_of_particles_);
        attribute[1].write(H5::PredType::NATIVE_INT,&tmp_number_ofspecies);
        attribute[2].write(H5::PredType::NATIVE_INT,&tmp_ring_size);
        attribute[3].write(H5::PredType::NATIVE_INT,&tmp_critical_ring_size);
        attribute[4].write(H5::PredType::NATIVE_DOUBLE,&activation_rate_);
    }

    /// Create threads for an ens simulation
    /// @param number_of_ens number of threads
    /// @param results vector of vectors containing the individual results
    virtual void CreateThreads(int number_of_ens,std::vector<std::vector<double>>& results)=0;

    
    /// perform ensemble av.
    /// @param number_of_ens number of ensembles
    /// @param group_name name of the group containing the results
    /// @param file hdf5 file for storage
    double GetEnsAv(int number_of_ens,std::string group_name,H5::H5File file){
        std::vector<std::vector<double>> results(number_of_ens);
        
        CreateThreads(number_of_ens,results);
        
        long max_size=0;
        for (auto&x: results) {
            max_size=max_size<x.size()?x.size():max_size;
        }
        std::vector<double> ens_av_time_trace(2*max_size,0);
        for (auto&x: results) {
            for (long i=0; i<max_size; ++i) {
                ens_av_time_trace[2*i]+=time_between_recordings_*i;
                if (i<x.size()) {
                    ens_av_time_trace[2*i+1]+=x[i];
                }
                else{
                    ens_av_time_trace[2*i+1]+=x.back();
                }
            }
        }
        
        double number_of_ens_double=static_cast<double>(number_of_ens);
        for (auto& x:ens_av_time_trace) {
            x/=number_of_ens_double;
        }
        
        if (record_time_evolution_) {
            hsize_t dims[2];
            dims[0]=max_size;
            dims[1]=2;
            H5::Group group_time_trace=file.createGroup(group_name.data());
            H5::DataSpace dataspace(2,dims);
            H5::DataSet dataset=group_time_trace.createDataSet("time_trace", H5::PredType::NATIVE_DOUBLE, dataspace);
            dataset.write(ens_av_time_trace.data(), H5::PredType::NATIVE_DOUBLE);
            WriteReactionAndDecayParametersToHDF5Group(group_time_trace);
            AttachAttributesToHDF5Group(group_time_trace);
        }
        return ens_av_time_trace.back();
        
    }

protected:
    
    //member variables containing all information about the system
    bool record_time_evolution_;
    
    double time_between_recordings_;
    double next_recording_;
    double time_;
    
    double activation_rate_;
    double accumulated_decay_rate_;
    
    long initial_number_of_particles_;
    long maximal_number_of_ring_structures_;
    long decay_structures_;
    
    std::array<double, ring_size-1> reaction_rates_;
    std::array<double, critical_size-2> decay_rates_;
    
    std::array<double, 1+2*number_of_species> accumulated_rates_;
    std::vector<double> time_trace_;
    
    std::mt19937 rng_engine_;
};

//
template <int ring_size,int number_of_species,int critical_size>
/// specification for the heterogeneous system
class RingStructure: public  RingStructureBaseClass<ring_size, number_of_species, critical_size> {
    
public:

    /// constructor
    /// @param initial_number_of_particles initial number of particle in the inactive state
    /// @param activation_rate activation rate
    /// @param reaction_rates binding rates for all different polymer sizes
    /// @param decay_rates decay rates for all subcritical ploymers
    /// @param record_evolution bool for recording of intermediate states
    /// @param time_between_recordings time between recordings
    /// @param seed seed for the random number generator (has nothing to do with nucleation)
    RingStructure(long initial_number_of_particles,double activation_rate,std::array<double, ring_size-1> reaction_rates,std::array<double, critical_size-2> decay_rates,bool record_evolution,double time_between_recordings,long seed)://Begin List
    RingStructureBaseClass<ring_size,number_of_species,critical_size>(initial_number_of_particles,activation_rate,reaction_rates,decay_rates, record_evolution, time_between_recordings,seed)
    {//Begin Function
        for(auto& x:inactive_particles_){
            x=initial_number_of_particles;
        }
        for(auto& x:ring_structures_){
            for(auto& y:x){
                y=0;
            }
        }
        
        for (auto& x: rates_monomer_) {
            x=0;
        }
        
    };//End Constructor
    
    
    
    /// Copy constructor
    /// @param original other system
    /// @param seed seed for the random number generator for different time evolution (has nothing to do with nucleation)
    RingStructure(const RingStructure& original, long seed):
    RingStructureBaseClass<ring_size,number_of_species,critical_size>(original,seed),
    inactive_particles_{original.inactive_particles_},
    ring_structures_{original.ring_structures_},
    rates_monomer_{original.rates_monomer_}
    {}//End Copy Constructor
    
    
    /// get left binding partner of a monomer
    /// @param monomer monomer index
    /// @return binding partner index
    inline const int GetLeftPartner(const int monomer)const{
        return monomer?monomer-1:number_of_species-1;
    }
    
    /// get right binding partner of a monomer
    /// @param monomer monomer index
    /// @return binding partner index
    inline const int GetRightPartner(const int monomer)const{
        return monomer==number_of_species-1?0:monomer+1;
    }

    
    /// get right binding partner of a monomer
    /// @param left_end left end of the polymer
    /// @param size size of the ploymer
    /// @return monomer index
    inline const int GetRightPartnerPolymer(const int left_end,const int size)const{
        return (left_end+size+1)%number_of_species;
    }

    
    /// add monomer to ploymer
    /// @param left_end polymer left end
    /// @param size polymer size
    void AddParticle(const int left_end,const int size){//Begin Function
        
        if (size) {//no Monomer
            if (size<ring_size-1) {//not finished
                if (size<critical_size-1) {//subcrit
                
                    this->accumulated_decay_rate_+=this->decay_rates_[size-1];
                    ++(this->decay_structures_);
                
                }//subcrit
            
                //increase reaction rate left
                rates_monomer_[GetLeftPartner(left_end)]+=(this->reaction_rates_)[size]*ring_structures_[GetLeftPartner(left_end)][0];
                //increase reaction rate right
                rates_monomer_[GetRightPartnerPolymer(left_end, size)]+=(this->reaction_rates_)[size]*ring_structures_[GetRightPartnerPolymer(left_end, size)][0];
                
            }//not finished
            
        }//no Monomer
        
        
        else{//Monomer
            
            //get rate for left reaction
            if (rates_monomer_[left_end]) {//if there is reference for rate left
                rates_monomer_[left_end]+=rates_monomer_[left_end]/(static_cast<double>(ring_structures_[left_end][0]));
            }
            
            else{//get reference for rate left
                
                int element_to_the_left=GetLeftPartner(left_end);
                for (int i=0; i<number_of_species; ++i) {//left end
                    for (int j=element_to_the_left-i<0?element_to_the_left-i+number_of_species:element_to_the_left-i; j<ring_size-1; j+=number_of_species) {//right end
                            rates_monomer_[left_end]+=((this->reaction_rates_)[j]*ring_structures_[i][j]);
                    }//right end
                }//left end
                
                //get rate for right reaction polymer
                for (int i=1; i<ring_size-1; ++i) {
                    rates_monomer_[left_end]+=(this->reaction_rates_[i])*ring_structures_[GetRightPartner(left_end)][i];
                }

                
            }//get rate for left reaction
            
            //get rate for right reaction monomer
            rates_monomer_[GetRightPartner(left_end)]+=ring_structures_[GetRightPartner(left_end)][0]*(this->reaction_rates_)[0];
            
            
        }//Monomer
        
        //increase number of ring structures
        ++ring_structures_[left_end][size];
        
    }//End AddParticle
    
    
    /// remove particle from polymer
    /// @param left_end polymer left end
    /// @param size polymer size
    void RemoveParticle(const int left_end,const int size){//Begin Function
        
        if (size) {//no Monomer
            
                if (size<ring_size-1) {//not finished
                    
                    if (size<critical_size-1) {//subcrit
                        
                            if (--(this->decay_structures_)) {//avoid accumulation of roundings
                                (this->accumulated_decay_rate_)-=(this->decay_rates_)[size-1];
                            }
                            else{
                                (this->accumulated_decay_rate_)=0;
                            }//avoid accumulation of roundings
                        
                
                    }//subcrit
            
                    //decrease reaction rate left
                    rates_monomer_[GetLeftPartner(left_end)]-=(this->reaction_rates_)[size]*ring_structures_[GetLeftPartner(left_end)][0];
                    
                    //decrease reaction rate right
                    rates_monomer_[GetRightPartnerPolymer(left_end, size)]-=(this->reaction_rates_)[size]*ring_structures_[GetRightPartnerPolymer(left_end, size)][0];
                    
                }//not finished
            
        }//no Monomer
        
        else{//Monomer
            
            //get rate for left reaction
            if (ring_structures_[left_end][0]==1) {//last monomer
                rates_monomer_[left_end]=0;
            }
            else {//other monomers left
                rates_monomer_[left_end]-=rates_monomer_[left_end]/(static_cast<double>(ring_structures_[left_end][0]));
            }
            
            //get rate for right reaction monomer
            rates_monomer_[GetRightPartner(left_end)]-=ring_structures_[GetRightPartner(left_end)][0]*(this->reaction_rates_)[0];
            
            
            
        }//Monomer
        
        //decrease number of ring structures
        --ring_structures_[left_end][size];
        
    }//End RemoveParticle
    
    
    /// activate inactive partivle
    /// @param species species to activate
    void ActivateParticle(const int species){
        --inactive_particles_[species];
        AddParticle(species, 0);
        
    }
    
    /// Combine a monomer with a polymer (reaction)
    /// @param monomer monomer index
    /// @param left_end polymer left end
    /// @param size size of the polymer
    void CombineMonomerAndRingStructure(const int monomer,const int left_end,const int size){//Begin Function
        
        //Remove particles which are used
        RemoveParticle(monomer, 0);

        RemoveParticle(left_end, size);
        
        //Find particle which will be created
        if (monomer==GetLeftPartner(left_end)&&monomer==GetRightPartnerPolymer(left_end, size)) {//Monomer can bind at both ends
            
            switch (std::uniform_int_distribution<int>(1,2)(this->rng_engine_)) {//random choice
                case 1:
                    AddParticle(monomer, size+1);
                    break;
                case 2:
                    AddParticle(left_end, size+1);
                    break;
                default:
                    //should not happen
                    std::cout<<"Random choice of binding failed";
                    exit(1);
            }//random choice
            
        }//Monomer can bind at both ends
        
        else if(monomer==GetLeftPartner(left_end)){//Monomer binds left
            AddParticle(monomer, size+1);
        }
        else{//Monomer binds right
            AddParticle(left_end, size+1);
        }
        
    }// End CombineMonomerAndRingStructure
    
    
    /// destroys a subcritical polymer
    /// @param left_end left end of the polymer
    /// @param size size of the polymer
    void DestroySeed(int left_end, const int size){
        
        //remove the particle
        RemoveParticle(left_end, size);
        
        //add respective monomers
        for (int i=0; i<size+1; ++i) {
            AddParticle(left_end, 0);
            left_end=GetRightPartner(left_end);
        }
        
    }
    
    /// destroys a random subcritical polymer
    void DestroyRandomSeed(){
        
        double random_number=std::uniform_real_distribution<double>(0,(this->accumulated_decay_rate_))(this->rng_engine_);
        double accumulated_value=0;
        
        for (int i=0; accumulated_value<random_number; ++i) {//left end
            
            for (int j=1; j<critical_size-1; ++j) {//right end
                
                accumulated_value+=ring_structures_[i][j]*(this->decay_rates_)[j-1];
                if (random_number<accumulated_value) {//element found
                    DestroySeed(i, j);
                    break;
                }//element found
            }//right end
            
        }//left end
        
    }
    
    /// calculate all rates for Gillespie
    void GetAccumulatedRates(){
        //first element is different (activate particle 0)
        (this->accumulated_rates_)[0]=inactive_particles_[0]*(this->activation_rate_);
        for (int i=1; i<number_of_species; ++i) {//activate particle
            (this->accumulated_rates_)[i]=(this->accumulated_rates_)[i-1]+inactive_particles_[i]*(this->activation_rate_);
        }//activate particle
        
        for (int i=0; i<number_of_species; ++i) {//reaction
            (this->accumulated_rates_)[i+number_of_species]=(this->accumulated_rates_)[i-1+number_of_species]+rates_monomer_[i];
        }//reaction
        
        //decay
        (this->accumulated_rates_)[2*number_of_species]=(this->accumulated_rates_)[2*number_of_species-1]+(this->accumulated_decay_rate_);
    }
    
    /// record all information for later storage
    void RecordTimeTrace(){
        long finished_structures=0;
        
        for (int i=0; i<number_of_species; ++i) {
            finished_structures+=ring_structures_[i][ring_size-1];
        }
        while ((this->next_recording_)<(this->time_)) {
            (this->time_trace_).push_back(static_cast<double>(finished_structures)/static_cast<double>((this->maximal_number_of_ring_structures_)));
            (this->next_recording_)+=(this->time_between_recordings_);
        }
    }
    
    /// perform a binding event
    /// @param monomer monomer index
    void EventMonomerBinds(const int monomer){
        double random_number=std::uniform_real_distribution<double>(0,rates_monomer_[monomer])((this->rng_engine_));
        double accumulated_value=0;
        
        //get rate for right reaction
        int element_to_the_right=GetRightPartner(monomer);
        for (int i=1; i<ring_size-1; ++i) {//length
            accumulated_value+=(this->reaction_rates_)[i]*ring_structures_[element_to_the_right][i]*ring_structures_[monomer][0];
            if (random_number<accumulated_value) {//element found
                CombineMonomerAndRingStructure(monomer, element_to_the_right, i);
                return;
            }//element found
        }//length
        
        //get rate for left reaction
        int element_to_the_left=GetLeftPartner(monomer);
        for (int i=0; i<number_of_species; ++i) {//left end
            for (int j=element_to_the_left-i<0?element_to_the_left-i+number_of_species:element_to_the_left-i; j<ring_size-1; j+=number_of_species) {//right end
                accumulated_value+=(this->reaction_rates_)[j]*ring_structures_[i][j]*ring_structures_[monomer][0];
                if (random_number<accumulated_value) {//element found
                    CombineMonomerAndRingStructure(monomer, i, j);
                    return;
                }//element found
            }//right end
        }//left end
        
        //should never get here...
        std::cout<<"Fatal error in EventMonomerBinds!\n";
        std::cout<<random_number<<"\n";
        std::cout<<accumulated_value<<"\n";

        std::raise(SIGTERM);
        exit(SIGTERM);
    }
    
    /// choose event for Gillespie step
    void ChooseEvent(){
        double random_number=std::uniform_real_distribution<double>(0,(this->accumulated_rates_).back())((this->rng_engine_));
        int event_index=0;
        
        while ((this->accumulated_rates_)[event_index]<random_number) {//get event
            ++event_index;
        }//get event
        
        if (event_index<number_of_species) {//activate particle
            ActivateParticle(event_index);
        }//activate particle
        
        else if(event_index<2*number_of_species){//reaction
            EventMonomerBinds(event_index-number_of_species);
        }//reaction
        
        else{//destroy polymer
            DestroyRandomSeed();
        }//destroy polymer
        
    }
    
    
    /// perform simulation
    void Simulate(){
        
        while ((this->IterationStep)()) {}
        
        long finished_structures=0;
        for (int i=0; i<number_of_species; ++i) {
            finished_structures+=ring_structures_[i][ring_size-1];
        }
        
        (this->time_trace_).push_back(static_cast<double>(finished_structures)/static_cast<double>((this->maximal_number_of_ring_structures_)));
    }

    
    /// perform simulation for a system copy
    /// @param seed for the random number generator for different time evolution (has nothing to do with nucleation)
    /// @param time_trace pointer to vector for storage
    void operator()(unsigned seed,std::vector<double>* time_trace){
        RingStructure<ring_size,number_of_species,critical_size> copy(*this,seed);
        copy.Simulate();
        copy.GetResults(time_trace);
    }
    
    /// Create threads for an ens simulation
    /// @param number_of_ens number of threads
    /// @param results vector of vectors containing the individual results
    void CreateThreads(int number_of_ens,std::vector<std::vector<double>>& results){
        std::vector<std::thread> threads;
        //create different seeds for all threads
        auto random_init=std::bind(std::uniform_int_distribution<unsigned>(0,~(0u)), std::ref(this->rng_engine_));
        //create all threads
        for (int i=0; i<number_of_ens; ++i) {
            threads.push_back(std::thread{*this,random_init(),results.data()+i});
            
        }
        //wait for all threads to finish
        for (std::thread& t:threads) {
            if (t.joinable()) {
                t.join();            }
            
        }
    }
    
private:
    std::array<long, number_of_species> inactive_particles_;
    std::array<std::array<long,ring_size>,number_of_species> ring_structures_;
    std::array<double, number_of_species> rates_monomer_;
};//class RingStructure




//homogeneous case (can be simulated much faster)
template <int ring_size,int critical_size>
class RingStructure<ring_size,1,critical_size>: public  RingStructureBaseClass<ring_size, 1, critical_size>{
    
public:
    
    ///Constructor
    ///@param initial_number_of_particles
    ///@param activation_rate
    ///@param reaction_rates
    ///@param decay_rates
    ///@param record_evolution
    ///@param time_between_recordings
    ///@param seed
    RingStructure(const long initial_number_of_particles, const double activation_rate, const std::array<double, ring_size-1> reaction_rates, const std::array<double, critical_size-2> decay_rates, const bool record_evolution, const double time_between_recordings, const long seed)://Begin List
    RingStructureBaseClass<ring_size, 1, critical_size>(initial_number_of_particles,activation_rate,reaction_rates,decay_rates,record_evolution,time_between_recordings,seed)
    {//Begin Function
        inactive_particles_=initial_number_of_particles;
        for(auto& x:ring_structures_){
            x=0;
        }
        
        rates_monomer_=0;
        
    };//End Constructor

    ///Copy Constructor
    ///@param RingStructure
    ///@param Seed
    RingStructure(const RingStructure& original, const long seed):
    RingStructureBaseClass<ring_size,1,critical_size>(original,seed),
    inactive_particles_{original.inactive_particles_},
    ring_structures_{original.ring_structures_},
    rates_monomer_{original.rates_monomer_}
    {}//End Copy Constructor
    
    // add monomer to ploymer
    /// @param size polymer size
    void AddParticle(const int size){//Begin Function
        
        if (size) {//no Monomer
            if (size<ring_size-1) {//not finished
                
                if (size<critical_size-1) {//subcrit
                    
                    (this->accumulated_decay_rate_)+=(this->decay_rates_)[size-1];
                    ++(this->decay_structures_);
                    
                }//subcrit
                
                //increase reaction rate left & right
                rates_monomer_+=2.0*(this->reaction_rates_)[size]*ring_structures_[0];
                
            }//not finished
            
        }//no Monomer
        
        
        else{//Monomer
            
            
            if (ring_structures_[0]) {
                rates_monomer_+=rates_monomer_/(static_cast<double>(ring_structures_[0]));
                rates_monomer_+=static_cast<double>(ring_structures_[0]+1)*(this->reaction_rates_)[0];
                
            }
            
            else{
                for (int j=0; j<ring_size-1; ++j) {
                    rates_monomer_+=2.0*(this->reaction_rates_)[j]*ring_structures_[j];
                }
                
                
            }
            
            
        }//Monomer
        
        //increase number of ring structures
        ++ring_structures_[size];
        
    }//End AddParticle
    
    
    /// remove a ploymer from  the system
    /// @param size polymer size
    void RemoveParticle(const int size){//Begin Function
        
        if (size) {//no Monomer
            if (size<ring_size-1) {//not finished
                
                if (size<critical_size-1) {//subcrit
                    if (--(this->decay_structures_)) {
                        (this->accumulated_decay_rate_)-=(this->decay_rates_)[size-1];
                    }
                    else{
                        (this->accumulated_decay_rate_)=0;
                    }
                    
                }//subcrit
                
                //decrease reaction rate left & right
                rates_monomer_-=2.0*(this->reaction_rates_)[size]*ring_structures_[0];
                
            }//not finished
            
        }//no Monomer
        
        
        else{//Monomer
            
            if (ring_structures_[0]==1) {
                rates_monomer_=0;
            }
            else{
                rates_monomer_-=rates_monomer_/(static_cast<double>(ring_structures_[0]));
                rates_monomer_-=static_cast<double>(ring_structures_[0]-1)*(this->reaction_rates_)[0];
            }
            
        }//Monomer
        
        //decrease number of ring structures
        --ring_structures_[size];
        
    }//End RemoveParticle
    
    
    /// activate a particle
    void ActivateParticle(){
        --inactive_particles_;
        AddParticle(0);
    }
    
    /// combines a monomer with a polymer
    /// @param size size of the polymer
    void CombineMonomerAndRingStructure(const int size){//Begin Function
        
        //Remove particles which are used
        RemoveParticle(0);

        RemoveParticle(size);
        
        AddParticle(size+1);
        
        
    }// End CombineMonomerAndRingStructure
    
    
    /// destroy a seed
    /// @param size size of the seed
    void DestroySeed(const int size){
        
        RemoveParticle(size);
        
        for (int i=0; i<size+1; ++i) {
            AddParticle(0);
        }
        
    }
    
    /// destroy a random ploymer
    void DestroyRandomSeed(){
        
        double random_number=std::uniform_real_distribution<double>(0,this->accumulated_decay_rate_)(this->rng_engine_);
        double accumulated_value=0;
        
        
        for (int j=1; j<critical_size-1; ++j) {
            
            accumulated_value+=ring_structures_[j]*(this->decay_rates_)[j-1];
            if (random_number<accumulated_value) {
                DestroySeed(j);
                break;
            }
        }
        
    }
    
    /// calculate all rates for Gillespie
    void GetAccumulatedRates(){
        (this->accumulated_rates_)[0]=inactive_particles_*(this->activation_rate_);
        (this->accumulated_rates_)[1]=(this->accumulated_rates_)[0]+rates_monomer_;
        (this->accumulated_rates_)[2]=(this->accumulated_rates_)[1]+(this->accumulated_decay_rate_);
    }
    
    /// update time according to gillespie
    inline void UpdateTime(){
        (this->time_)+=std::exponential_distribution<double>((this->accumulated_rates_).back())(this->rng_engine_);
    }
    
    /// perform monomer binding event
    void EventMonomerBinds(){
        double random_number=std::uniform_real_distribution<double>(0,rates_monomer_)(this->rng_engine_);
        double accumulated_value=0;
        
        accumulated_value+=(this->reaction_rates_)[0]*(ring_structures_[0]-1)*ring_structures_[0];
        if (random_number<accumulated_value) {
            CombineMonomerAndRingStructure(0);
            return;
        }
        for (int i=1; i<ring_size-1; ++i) {
            accumulated_value+=2.0*(this->reaction_rates_)[i]*ring_structures_[i]*ring_structures_[0];
            if (random_number<accumulated_value) {
                CombineMonomerAndRingStructure(i);
                return;
            }
        }
        std::cout<<"Fatal error in EventMonomerBinds!\n";
        std::cout<<random_number<<"\n";
        std::cout<<accumulated_value<<"\n";
        
//        std::raise(SIGTERM);
        exit(SIGTERM);
    }
    
    /// choose event according to Gillespie
    void ChooseEvent(){
        double random_number=std::uniform_real_distribution<double>(0,(this->accumulated_rates_).back())(this->rng_engine_);
        if (random_number<(this->accumulated_rates_)[0]) {
            ActivateParticle();
        }
        else if(random_number<(this->accumulated_rates_)[1]){
            EventMonomerBinds();
        }
        else{
            DestroyRandomSeed();
        }
    }
    
    /// make all recordings
    void RecordTimeTrace(){
        double yield=static_cast<double>(ring_structures_[ring_size-1])/static_cast<double>((this->maximal_number_of_ring_structures_));
                while ((this->next_recording_)<(this->time_)) {
                    (this->time_trace_).push_back(yield);
                    (this->next_recording_)+=(this->time_between_recordings_);
                }
    }
    
    /// perform iteration step
    /// @return bool deciding if the simulation has finished
    const bool IterationStep(){
        
        GetAccumulatedRates();
        
        if (0.0<(this->accumulated_rates_).back()) {//Reaction can takes place
            UpdateTime();
            if (this->record_time_evolution_&&this->time_>this->next_recording_) {
                RecordTimeTrace();
            }
            ChooseEvent();
            return true;
        }
        
        else return false;//final state
    }
    
    /// perform simulation
    void Simulate(){
        while (IterationStep()) {}
        (this->time_trace_).push_back(static_cast<double>(ring_structures_[ring_size-1])/static_cast<double>(this->maximal_number_of_ring_structures_));
        
    }

    /// perform simulation for a system copy
    /// @param seed for the random number generator for different time evolution (has nothing to do with nucleation)
    /// @param time_trace pointer to vector for storage
    void operator()(unsigned seed,std::vector<double>* time_trace){
        RingStructure<ring_size,1,critical_size> copy(*this,seed);
        copy.Simulate();
        copy.GetResults(time_trace);
    }
    
    /// Create threads for an ens simulation
    /// @param number_of_ens number of threads
    /// @param results vector of vectors containing the individual results
    void CreateThreads(int number_of_ens,std::vector<std::vector<double>>& results){
        std::vector<std::thread> threads;
        //create different seeds for all threads
        auto random_init=std::bind(std::uniform_int_distribution<unsigned>(0,~(0u)), std::ref(this->rng_engine_));
        //create all threads
        for (int i=0; i<number_of_ens; ++i) {
            threads.push_back(std::thread{*this,random_init(),results.data()+i});
        }
        //wait for all threads to finish
        for (std::thread& t:threads) {
            if (t.joinable()) {
                t.join();            }
            
        }
    }

private:
    
    //reduced number of member variables for homogeneous case
    long inactive_particles_;
    std::array<long,ring_size> ring_structures_;
    double rates_monomer_;
    

};




#endif /* system_class_h */
